<?php
/**
 * Plugin Name: FluentAffiliate - Connector
 * Description: Connect this site with your primary store domain
 * Version: 1.0.1
 * Author: WPManageNinja
 * Text Domain: fluent-affiliate-connector
 */

// Exit if accessed directly

if (!defined('ABSPATH')) {
    exit;
}

define('FLUENT_AFFILIATE_CONNECTOR_VERSION', '1.0.1');
define('FLUENT_AFFILIATE_CONNECTOR_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('FLUENT_AFFILIATE_CONNECTOR_PLUGIN_URL', plugin_dir_url(__FILE__));
define('FLUENT_AFFILIATE_CONNECTOR_DIR_FILE', __FILE__);

class FluentAffiliateConnectorBootstrap
{
    private $connector = null;

    public function __construct()
    {
        require_once FLUENT_AFFILIATE_CONNECTOR_PLUGIN_PATH . 'Classes/Connector.php';
        $this->connector = new \FluentAffiliateConnector\Classes\Connector();
        $this->connector->register();

        add_action('admin_menu', array($this, 'admin_menu'));
        add_action('wp_ajax_fluent-affiliate-connector-settings', array($this, 'getSettings'));
        add_action('wp_ajax_fluent-affiliate-connector-save-token', array($this, 'saveToken'));
        add_action('wp_ajax_fluent-affiliate-connector-save-settings', array($this, 'saveSettings'));

        add_action('wp_enqueue_scripts', [$this, 'loadTrackerJs'], 9999);

        add_action('wp_ajax_fluent_aff_connector_count_visit', [$this, 'processVisit']);
        add_action('wp_ajax_nopriv_fluent_aff_connector_count_visit', [$this, 'processVisit']);
    }

    public function admin_menu()
    {
        add_submenu_page(
            'options-general.php',
            __('FluentAffiliate Connector', 'fluent-affiliate-connect'),
            __('FluentAffiliate Connector', 'fluent-affiliate-connect'),
            'manage_options',
            'fluent-affiliate-client',
            array($this, 'settingsPage'),
            100
        );
    }

    public function settingsPage()
    {
        wp_enqueue_script('fluent-affiliate-connect-script', FLUENT_AFFILIATE_CONNECTOR_PLUGIN_URL . 'dist/app.js', array('jquery'), FLUENT_AFFILIATE_CONNECTOR_VERSION, true);
        wp_localize_script('fluent-affiliate-connect-script', 'fluentAffiliateConnectVars', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce('fluent_affiliate_connect_nonce'),
            'version'  => FLUENT_AFFILIATE_CONNECTOR_VERSION
        ));
        ?>
        <div class="wrap">
            <div id="fluent_app"></div>
        </div>
        <?php
    }

    public function getSettings()
    {
        $this->verifyAjaxRequest();
        $settings = $this->connector->getSettings();

        if (!empty($settings['server_token'])) {
            $settings['server_token'] = '**********';
        }

        $clientConfig = json_encode([
            'callback_url' => admin_url('admin-ajax.php?action=fluent-affiliate-connect-callback'),
            'site_url'     => site_url(),
            'site_title'   => get_bloginfo('name'),
        ], JSON_UNESCAPED_SLASHES);

        wp_send_json([
            'settings'      => $settings,
            'client_config' => $clientConfig
        ], 200);
    }

    public function saveToken()
    {
        $this->verifyAjaxRequest();

        $token = isset($_REQUEST['token']) ? trim($_REQUEST['token']) : '';

        $token = wp_unslash($token);

        $config = json_decode($token, true);

        if (!$config || empty($config['server_token']) || empty($config['callback_url']) || empty($config['store_url']) || empty($config['site_id']) || empty($config['param_key'])) {
            wp_send_json(array('message' => __('Invalid token.', 'fluent-affiliate-connect')), 400);
        }

        $settings = $this->connector->getSettings();
        $settings['store_url'] = sanitize_url($config['store_url']);
        $settings['callback_url'] = sanitize_url($config['callback_url']);
        $settings['site_id'] = sanitize_text_field($config['site_id']);
        $settings['server_token'] = sanitize_textarea_field($config['server_token']);
        $settings['param_key'] = sanitize_text_field($config['param_key']);
        $settings['cookie_duration'] = sanitize_text_field($config['cookie_duration']);
        $settings['credit_last_referrer'] = sanitize_text_field($config['credit_last_referrer']);
        $settings['status'] = 'yes';

        update_option('_fluent_affiliate_connect_settings', $settings, 'yes');

        wp_send_json(array(
            'message' => __('Your site has been connected successfully.', 'fluent-affiliate-connect'),
        ), 200);

    }

    public function saveSettings()
    {
        $this->verifyAjaxRequest();

        $settings = !empty($_REQUEST['settings']) ? $_REQUEST['settings'] : [];

        if (!empty($settings['disconnect']) && $settings['disconnect'] == 'yes') {
            delete_option('_fluent_affiliate_connect_settings');

            wp_send_json(array(
                'message' => __('Your site has been disconnected successfully.', 'fluent-affiliate-connect'),
            ), 200);
        }

        $prevSettings = $this->connector->getSettings();

        update_option('_fluent_affiliate_connect_settings', $prevSettings, 'yes');

        wp_send_json(array(
            'message' => __('Settings saved successfully.', 'fluent-affiliate-connect'),
        ), 200);
    }

    private function verifyAjaxRequest()
    {
        if (!current_user_can('manage_options')) {
            wp_send_json(array('message' => __('You do not have permission to do this action.', 'fluent-affiliate-connect')), 403);
        }

        $nonce = isset($_REQUEST['__nonce']) ? sanitize_text_field($_REQUEST['__nonce']) : '';
        if (!wp_verify_nonce($nonce, 'fluent_affiliate_connect_nonce')) {
            wp_send_json(array('message' => __('Invalid nonce.', 'fluent_affiliate_connect_nonce')), 403);
        }
    }

    public function loadTrackerJs()
    {
        if (!apply_filters('fluent_affiliate/will_load_tracker_js', true)) {
            return;
        }

        $settings = $this->connector->getSettings();
        if ($settings['status'] != 'yes') {
            return;
        }

        $storeUrl = $settings['store_url'] ?? '';

        // remove trailing slash from store URL
        if (substr($storeUrl, -1) === '/') {
            $storeUrl = rtrim($storeUrl, '/');
        }

        wp_enqueue_script('fluent_aff_connector', FLUENT_AFFILIATE_CONNECTOR_PLUGIN_URL . 'dist/fluent_aff.js', [], FLUENT_AFFILIATE_CONNECTOR_VERSION, true);
        wp_localize_script('fluent_aff_connector', 'fluent_aff_connect', apply_filters('fluent_affiliate_connector_tracker_vars', [
            'duration_days' => $settings['cookie_duration'] ?? 30,
            'aff_param'     => $settings['param_key'] ?? 'ref',
            'other_params'  => [
                'utm_source',
                'utm_medium',
                'utm_campaign'
            ],
            'auto_params'   => [
                'utm_source',
                'utm_medium',
                'utm_campaign'
            ],
            'store_url'     => $storeUrl,
            'credit_last'   => $settings['credit_last_referrer'] === 'yes' ? 1 : '',
            'request_url'   => admin_url('admin-ajax.php?action=fluent_aff_connector_count_visit'),
        ]));
    }

    public function processVisit()
    {
        $payloadData = array_filter([
            'aff_id'       => isset($_POST['__aff_id__']) ? sanitize_text_field($_POST['__aff_id__']) : '',
            'referrer'     => isset($_POST['refer']) ? sanitize_url($_POST['refer']) : '',
            'url'          => isset($_POST['url']) ? sanitize_url($_POST['url']) : '',
            'utm_campaign' => isset($_POST['utm_campaign']) ? sanitize_text_field($_POST['utm_campaign']) : '',
            'utm_source'   => isset($_POST['utm_source']) ? sanitize_text_field($_POST['utm_source']) : '',
            'utm_medium'   => isset($_POST['utm_medium']) ? sanitize_text_field($_POST['utm_medium']) : '',
            'ip_address'   => $this->connector->getIp(false)
        ]);

        if (empty($payloadData['aff_id'])) {
            wp_send_json([
                'message' => 'invalid_param',
                'time'    => time()
            ]);
        }

        // let's send this data to the main server
        $settings = $this->connector->getSettings();

        $callBackUrl = $settings['callback_url'] ?? '';
        if (empty($callBackUrl)) {
            wp_send_json([
                'message' => 'callback_url_not_set',
                'time'    => time()
            ]);
        }

        $sendingData = [
            'server_token' => $settings['server_token'] ?? '',
            'site_id'      => $settings['site_id'] ?? '',
            'visit'        => $payloadData
        ];

        $response = wp_remote_post($callBackUrl, [
            'method'    => 'POST',
            'body'      => json_encode($sendingData),
            'timeout'   => 15,
            'sslverify' => false,
            'headers'   => [
                'Content-Type' => 'application/json',
            ],
        ]);


        $responseData = array_filter([
            '__aff_id__'   => $payloadData['aff_id'],
            'utm_campaign' => $payloadData['utm_campaign'] ?? '',
            'utm_source'   => $payloadData['utm_source'] ?? '',
            'utm_medium'   => $payloadData['utm_medium'] ?? ''
        ]);

        if (is_wp_error($response)) {
            wp_send_json([
                'response' => $responseData
            ]);
        }

        $body = wp_remote_retrieve_body($response);
        $body = json_decode($body, true);


        if (!$body || empty($body['visit_id'])) {
            wp_send_json([
                'response' => $responseData
            ]);
        }

        $responseData['visit_id'] = $body['visit_id'] ?? '';

        wp_send_json([
            'response' => $responseData
        ]);
    }
}


new FluentAffiliateConnectorBootstrap();
